#define F_CPU 16000000
#include <util/delay.h>
#include <inttypes.h>

#include "LiquidCrystal.h"


// When the display powers up, it is configured as follows:
//
// 1. Display clear
// 2. Function set:
//    DL = 1; 8-bit interface data
//    N = 0; 1-line display
//    F = 0; 5x8 dot character font
// 3. Display on/off control:
//    D = 0; Display off
//    C = 0; Cursor off
//    B = 0; Blinking off
// 4. Entry mode set:
//    I/D = 1; Increment by 1
//    S = 0; No shift
//
// Note, however, that resetting the Arduino doesn't reset the LCD, so we
// can't assume that its in that state when a sketch starts (and the
// LiquidCrystal constructor is called).


// LOW: write to LCD.  HIGH: read from LCD.
uint8_t _rw_mode; 

uint8_t _displayfunction;
uint8_t _displaycontrol;
uint8_t _displaymode;


/************ low level data pushing commands **********/

// low level private function
inline void write_command(uint8_t) __attribute__((always_inline)) ;
inline uint8_t write_data(uint8_t) __attribute__((always_inline)) ;
void send(uint8_t, uint8_t);
void write4bits(uint8_t);
void write8bits(uint8_t);
void pulseEnable();

void pulseEnable(void) {
	LCD_Ctrl_PORT &= ~(1 << LCD_E_PIN);
	_delay_us(1);
	LCD_Ctrl_PORT |= (1 << LCD_E_PIN);
	_delay_us(1);    // enable pulse must be >450ns
	LCD_Ctrl_PORT &= ~(1 << LCD_E_PIN);
	_delay_us(100);   // commands need > 37us to settle
}

void write4bits(uint8_t value) {

	// mettre les bits d4,d5,d6,d7 du port data en sortie
	LCD_Data_DDR |= 0xF0;

	// ecrir sur le nible high du  port data
	LCD_Data_PORT = (LCD_Data_PORT & 0xF) | (value << 4);
	pulseEnable();
}

void write8bits(uint8_t value) {
	// mettre le port date en sortie
	LCD_Data_DDR = 0xFF;
	// ecrir sur le nible high du  port data
	LCD_Data_PORT = value;

	pulseEnable();
}

// write either command or data, with automatic 4/8-bit selection
void send(uint8_t value, uint8_t mode) {

	if (mode) {
		LCD_Ctrl_PORT |= (1 << LCD_RS_PIN);
	}
	else {
		LCD_Ctrl_PORT &= ~(1 << LCD_RS_PIN);
	}
	// if there is a RW pin indicated, set it low to Write
	if (_rw_mode) {
		LCD_Ctrl_PORT &= ~(1 << LCD_RW_PIN);
	}

	if (_displayfunction & LCD_8BITMODE) {
		write8bits(value);
	}
	else {
		write4bits(value >> 4);
		write4bits(value);
	}
}

/*********** mid level commands, for sending data/cmds */

inline void write_command(uint8_t value){
	send(value, false);
}

inline uint8_t write_data(uint8_t value){
	send(value, true);
	return 1;
}

// par defaut lcd: matrice 5x8 et 8bits mode, R/W = LOW (lecture seul)
void lcd_begin() {
	lcd_beginMode(LCD_5x8DOTS, false, false);
}

void lcd_beginMode(uint8_t dotsize, uint8_t rw_mode, uint8_t fourbitmode) {

	_rw_mode = rw_mode;
	// declare RS, RW, E en sortie.
	LCD_Ctrl_DDR |= (1 << LCD_RS_PIN) | (1 << LCD_E_PIN);

	// we can save 1 pin by not using RW. Indicate by passing 255 to LCD_RW_PIN
	if (_rw_mode) {
		LCD_Ctrl_DDR |= (1 << LCD_RW_PIN);
	}
	else {
		LCD_Ctrl_DDR |= (1 << LCD_RW_PIN);
		LCD_Ctrl_PORT &= ~(1 << LCD_RW_PIN);
	}

	if (fourbitmode)
	_displayfunction = LCD_4BITMODE | LCD_1LINE | LCD_5x8DOTS;
	else
	_displayfunction = LCD_8BITMODE | LCD_1LINE | LCD_5x8DOTS;
	
	if (LCD_LINES > 1) {
		_displayfunction |= LCD_2LINE;
	}

	// for some 1 line displays you can select a 10 pixel high font
	if ((dotsize != 0) && (LCD_LINES == 1)) {
		_displayfunction |= LCD_5x10DOTS;
	}

	// SEE PAGE 45/46 FOR INITIALIZATION SPECIFICATION!
	// according to datasheet, we need at least 40ms after power rises above 2.7V
	// before sending commands. Arduino can turn on way befer 4.5V so we'll wait 50
	_delay_us(50000);
	// Now we pull both RS and R/W low to begin commands
	LCD_Ctrl_PORT &= ~((1 << LCD_RS_PIN) | (1 << LCD_E_PIN));
	if (_rw_mode) {
		LCD_Ctrl_PORT &= ~(1 << LCD_RW_PIN);
	}

	//put the LCD into 4 bit or 8 bit mode
	if (!(_displayfunction & LCD_8BITMODE)) {
		// this is according to the hitachi HD44780 datasheet
		// figure 24, pg 46

		// we start in 8bit mode, try to set 4 bit mode
		write4bits(0x03);
		_delay_us(4500); // wait min 4.1ms

		// second try
		write4bits(0x03);
		_delay_us(4500); // wait min 4.1ms

		// third go!
		write4bits(0x03);
		_delay_us(150);

		// finally, set to 4-bit interface
		write4bits(0x02);
	}
	else {
		// this is according to the hitachi HD44780 datasheet
		// page 45 figure 23

		// Send function set command sequence
		write_command(LCD_FUNCTIONSET | _displayfunction);
		_delay_us(4500);  // wait more than 4.1ms

		// second try
		write_command(LCD_FUNCTIONSET | _displayfunction);
		_delay_us(150);

		// third go
		write_command(LCD_FUNCTIONSET | _displayfunction);
	}

	// finally, set # lines, font size, etc.
	write_command(LCD_FUNCTIONSET | _displayfunction);

	// turn the display on with no cursor or blinking default
	_displaycontrol = LCD_DISPLAYON | LCD_CURSOROFF | LCD_BLINKOFF;
	lcd_display();

	// clear it off
	lcd_clear();

	// Initialize to default text direction (for romance languages)
	_displaymode = LCD_ENTRYLEFT | LCD_ENTRYSHIFTDECREMENT;
	// set the entry mode
	write_command(LCD_ENTRYMODESET | _displaymode);
}

/********** high level commands, for the user! */
void lcd_clear()
{
	write_command(LCD_CLEARDISPLAY);  // clear display, set cursor position to zero
	_delay_us(3000);  // this command takes a long time!
}

void lcd_clearln(uint8_t line)
{
	lcd_setCursor(0,line);
	for (uint8_t i=0; i < LCD_CHARS ;i++)
	{
		lcd_printChar(' ');
	}
}

void lcd_selfTest()
{
	volatile uint8_t line = 0;
	volatile uint8_t pos = 0;
	
	lcd_clear();
	lcd_setCursor(0,1);
	lcd_printStr("*** SELF TEST ***");
	lcd_clear();
	lcd_noDisplay();
	do {
		lcd_printChar(0x00);
		++pos;
		if (!(pos % LCD_CHARS)) {
			line++;
			lcd_setCursor(0, line);
		}
	} while (pos < (LCD_CHARS * (LCD_LINES)));
	
	for (uint8_t k = 0; k < 2; k++)
	{
		lcd_noDisplay();
		_delay_ms(250);
		lcd_display();
		_delay_ms(250);
	}
	lcd_clear();
	line = 0;
	pos = 0;
	do {
		lcd_printChar((char)pos);
		++pos;
		if (!(pos % LCD_CHARS)) {
			line++;
			lcd_setCursor(0, line);
			_delay_ms(500);
		}
		if (line == LCD_LINES) line = 0;
		if (!(pos % (LCD_LINES * LCD_CHARS))) lcd_clear();
	} while (pos < 255);
	lcd_clear();
	lcd_printStr("*** SELF TEST OK ***");
	_delay_ms(1000);
	lcd_clear();
}



void lcd_home()
{
	write_command(LCD_RETURNHOME);  // set cursor position to zero
	_delay_us(2000);  // this command takes a long time!
}

void lcd_setCursor(uint8_t col, uint8_t row)
{
	uint8_t row_offsets[] = { 0x00, 0x40, LCD_CHARS, 0x40 + LCD_CHARS};
	write_command(LCD_SETDDRAMADDR | col | row_offsets[row]);
}

// Turn the display on/off (quickly)
void lcd_noDisplay() {
	_displaycontrol &= ~LCD_DISPLAYON;
	write_command(LCD_DISPLAYCONTROL | _displaycontrol);
}

void lcd_display() {
	_displaycontrol |= LCD_DISPLAYON;
	write_command(LCD_DISPLAYCONTROL | _displaycontrol);
}

// Turns the underline cursor on/off
void lcd_noCursor() {
	_displaycontrol &= ~LCD_CURSORON;
	write_command(LCD_DISPLAYCONTROL | _displaycontrol);
}

void lcd_cursor() {
	_displaycontrol |= LCD_CURSORON;
	write_command(LCD_DISPLAYCONTROL | _displaycontrol);
}

// Turn on and off the blinking cursor
void lcd_noBlink() {
	_displaycontrol &= ~LCD_BLINKON;
	write_command(LCD_DISPLAYCONTROL | _displaycontrol);
}

void lcd_blink() {
	_displaycontrol |= LCD_BLINKON;
	write_command(LCD_DISPLAYCONTROL | _displaycontrol);
}

// These commands scroll the display without changing the RAM
void lcd_scrollDisplayLeft(void) {
	write_command(LCD_CURSORSHIFT | LCD_DISPLAYMOVE | LCD_MOVELEFT);
}

void lcd_scrollDisplayRight(void) {
	write_command(LCD_CURSORSHIFT | LCD_DISPLAYMOVE | LCD_MOVERIGHT);
}

// This is for text that flows Left to Right
void lcd_leftToRight(void) {
	_displaymode |= LCD_ENTRYLEFT;
	write_command(LCD_ENTRYMODESET | _displaymode);
}

// This is for text that flows Right to Left
void lcd_rightToLeft(void) {
	_displaymode &= ~LCD_ENTRYLEFT;
	write_command(LCD_ENTRYMODESET | _displaymode);
}

// This will 'right justify' text from the cursor
void lcd_autoscroll(void) {
	_displaymode |= LCD_ENTRYSHIFTINCREMENT;
	write_command(LCD_ENTRYMODESET | _displaymode);
}

// This will 'left justify' text from the cursor
void lcd_noAutoscroll(void) {
	_displaymode &= ~LCD_ENTRYSHIFTINCREMENT;
	write_command(LCD_ENTRYMODESET | _displaymode);
}

// Allows us to fill the first 8 CGRAM locations
// with custom characters
void lcd_createChar(uint8_t location, uint8_t charmap[]) {
	location &= 0x7; // we only have 8 locations 0-7
	write_command(LCD_SETCGRAMADDR | (location << 3));
	for (uint8_t i = 0; i < 8; i++) {
		write_data(charmap[i]);
	}
}


// Public Methods //////////////////////////////////////////////////////////////

/* default implementation */

uint8_t lcd_printStr(const char str[])
{
	uint8_t n = 0;
	while (*str) {
		n += write_data(*str++);
	}
	return n;
}

uint8_t lcd_printChar(char c)
{
	return write_data(c);
}

void lcd_printBinByte(uint8_t byte)
{
	uint8_t c;
	lcd_printStr("0b");
	for(uint8_t i = 0; i < 8 ; ++i)	{
		c = byte & 0x80;
		if (c) lcd_printChar('1'); else lcd_printChar('0');
		byte = byte << 1;
	}
}
void lcd_printBinWord(uint16_t word)
{
	uint16_t c;
	lcd_printStr("0b");
	for(uint8_t i = 0; i < 16 ; ++i)	{
		c = word & 0x8000;
		if (c) lcd_printChar('1'); else lcd_printChar('0');
		word = word << 1;
	}
}

uint8_t printNumber(unsigned long n, uint8_t base) {
	char buf[8 * sizeof(long) + 1]; // Assumes 8-bit chars plus zero byte.
	char *str = &buf[sizeof(buf) - 1];

	*str = '\0';

	// prevent crash if called with base == 1
	if (base < 2) base = 10;

	do {
		unsigned long m = n;
		n /= base;
		char c = m - base * n;
		*--str = c < 10 ? c + '0' : c + 'A' - 10;
	} while(n);

	return lcd_printStr(str);
}

uint8_t lcd_printInt(int64_t n, uint8_t base)
{
	if (base == 0) {
		return write_data(n);
		} else if (base == 10) {
		if (n < 0) {
			uint8_t t = lcd_printChar('-');
			n = -n;
			return printNumber(n, 10) + t;
		}
		return printNumber(n, 10);
		} else {
		return printNumber(n, base);
	}
}

uint8_t lcd_printuInt(uint64_t n, uint8_t base)
{
	if (base == 0) return write_data(n);
	else return printNumber(n, base);
}



uint8_t lcd_printHex(uint64_t n, uint8_t bytes) {
	char buf[bytes + 3]; // Assumes 8-bit chars plus zero byte terminate. + "0x" prefix
	char *str = &buf[sizeof(buf) - 1]; // pointeur sur la fin du buffer
	*str = '\0';

	do {
		unsigned long m = n;
		n /= HEX;
		char c = m - HEX * n;
		*--str = c < 10 ? c + '0' : c + 'A' - 10;
	} while(--bytes);
	*--str = 'x';
	*--str = '0';
	return lcd_printStr(str);
}
